function probitVS_temp2(X, v, PI, Y, MCMC, temp, s, block, maxneigh)
% PROBITVS_TEMP2 Bayesian auxiliary variable model for probit 
% regression using joint update to {z,lambda} with covariate set 
% uncertainty. Based on pseudocode A1 in Holmes and Held (2006).
% PROBITVS_TEMP2 attempts to improve chain mixing by
% Metropolis-Hastings coupling of the original chain of interest with a
% tempered chain (with higher temperature temp) (Geyer 1991).
%
% References: 
% Holmes, C. and Held, L. (2006) "Bayesian auxiliary variable models for 
% binary and multinomial regression", Bayesian Analysis
% Geyer, C.J. (1991) "Markov chain Monte Carlo maximum likelihood", In:
% Computational Science and Statistics: Proceedings of the 23rd Symposium
% on the Interface
%
% n = number of individuals 
% p = number of variables
%
% X = nxp design matrix 
% v = prior covariance matrix for beta. v is a pxp matrix
%     It is assumed here that v is diagonal (for computations of v_gam)!
% PI = prior probability Pr(GAM_i = 1) (can be of size 1x1 or px1)
% Y = nx1 column vector of binary variables
% MCMC = number of MCMC samples
% We are assuming that the prior mean for beta is the zero vector
%
% temp = temperature for tempered Markov chain (p(X)^(1/temp))
% s = probability for swap attempt in each iteration -> 
%     Number of iterations before next swap attempt: x ~ Geo(s)=s(1-s)^x
%     Expected number of iterations before next swap attempt = (1-s)/s
%
% optional:
% block = (sparse) pxp matrix with entries <>0, if the corresponding 
%         two genes should be updated together
% maxneigh = scalar indicating up to how many second-order neighbours 
%         should be included in blocks (in addition to first-order nb.)
% If block and depth parameters are given, then GAM vector is updated by
% block proposal, otherwise by simple addition/deletion proposal dist.

t1 = 1;
t2 = temp;

n = size(X,1);
p = size(X,2);

if numel(PI)==1
    PI = ones(p,1).*PI;
end

%%%% INITIALISE RANDOM VARIABLES 
GAM1 = zeros(p,1);
for i = 1:p
    U = rand;
    if U <= PI(i)
        GAM1(i) = 1;
    end
end
GAM2 = zeros(p,1);
for i = 1:p
    U = rand;
    if U <= PI(i)
        GAM2(i) = 1;
    end
end
Z1 = zeros(n,1); Z2 = zeros(n,1);
for i = 1:n
    if Y(i) > 0 
        Z1(i) = normlt(0,1,0);
        Z2(i) = normlt(0,1,0);
    else
        Z1(i) = normrt(0,1,0);
        Z2(i) = normrt(0,1,0);
    end
end

%save in sparse matrix form (col1=rowID, col2=colID, col3=beta):
fid = fopen(strcat('beta.',num2str(p),'.',num2str(temp),'.txt'), 'a');
fprintf(fid, '%10d %10d %1.12e\n', [MCMC p 0]');    %store dimensions

%save alpha and acceptance indicator:
fidalpha = fopen(strcat('alpha.',num2str(p),'.',num2str(temp),'.txt'), 'a');
fidindic = fopen(strcat('indic.',num2str(p),'.',num2str(temp),'.txt'), 'a');

%%%% START MCMC
if (nargin == 7)
  for i = 1:MCMC
    %target distribution of interest:
    [Z1,beta1,GAM1,R1] = ZbetaGAM_update(X,Y,v,PI, Z1,GAM1,t1);    

    %tempered distribution:
    [Z2,beta2,GAM2,R2] = ZbetaGAM_update(X,Y,v,PI, Z2,GAM2,t2);
    
    %Metropolis-Hastings acceptance for swapping states of chains 1 and 2
    alpha = min(1, exp((t2-t1)/(2*t1*t2) * sum(R1.^2 - R2.^2)));
    
    indic = 0;
    Us = rand;
    if Us <= s
        U = rand;
        if U < alpha
            GAMtmp = GAM2; GAM2 = GAM1; GAM1 = GAMtmp;
            betatmp = beta2; beta2 = beta1; beta1 = betatmp;
            Ztmp = Z2; Z2 = Z1; Z1 = Ztmp;
            
            indic = 1;
        end
    end
    
    %save current beta and GAM values of chain of interest:
    [rowGAM,colGAM] = find(GAM1);
    fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta1]');
    
    fprintf(fidalpha, '%1.6f ', alpha);
    fprintf(fidindic, '%1.6f ', indic);
  end  
  
elseif (nargin == 9)
  for i = 1:MCMC
    %target distribution of interest:
    [Z1,beta1,GAM1,R1]=ZbetaGAM_update(X,Y,v,PI,Z1,GAM1,t1,block,maxneigh);
    
    %tempered distribution:
    [Z2,beta2,GAM2,R2]=ZbetaGAM_update(X,Y,v,PI,Z2,GAM2,t2,block,maxneigh);   
    
    %Metropolis-Hastings acceptance for swapping states of chains 1 and 2
    alpha = min(1, exp((t2-t1)/(2*t1*t2) * sum(R1.^2 - R2.^2)));
    
    indic = 0;
    Us = rand;
    if Us <= s
        U = rand;
        if U < alpha
            GAMtmp = GAM2; GAM2 = GAM1; GAM1 = GAMtmp;
            betatmp = beta2; beta2 = beta1; beta1 = betatmp;
            Ztmp = Z2; Z2 = Z1; Z1 = Ztmp;
            
            indic = 1;
        end
    end
    
    %save current beta and GAM values of chain of interest:
    [rowGAM,colGAM] = find(GAM1);
    fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta1]');
    
    fprintf(fidalpha, '%1.6f ', alpha);
    fprintf(fidindic, '%1.6f ', indic);
    
  end
else
  error('Wrong # of arguments to probitVS_temp2');
end

fclose(fid);
fclose(fidalpha);
fclose(fidindic);

% Manuela Zucknick, last update: 12-09-2006
