function probitVS(X, v, PI, Y, MCMC, block, maxneigh);
% PROBITVS Bayesian auxiliary variable model for variable selection in
% binary probit regression using joint update of {z,beta,GAM}
% Based on pseudocode A1 by Chris Holmes and Leonhard Held (2006)
%
% Reference: 
% Holmes, C. and Held, L. (2006) "Bayesian auxiliary variable models for 
% binary and multinomial regression", Bayesian Analysis 1:145-168
%
% n = number of individuals 
% p = number of variables
%
% X = nxp design matrix 
% v = prior covariance matrix for beta. v is a pxp matrix
% PI = prior probability Pr(GAM_i = 1)
% Y = nx1 column vector of binary variables
% MCMC = number of MCMC samples
% We are assuming that the prior mean for beta is the  zero (vector)
%
% optional:
% block = (sparse) pxp matrix with entries <>0, if the corresponding 
%         two genes should be updated together
% maxneigh = number of genes selected for updating among all first- and 
%            second-order neighbours
% If parameters block and maxneigh are given, then the GAM vector is 
% updated by a block proposal, otherwise by an addition/deletion proposal.

T = 1;  %no tempering here

n = size(X,1);
p = size(X,2);

if numel(PI)==1
    PI = ones(p,1).*PI;
end

%%%% INITIALISE RANDOM VARIABLES
GAM = zeros(p,1);
for i = 1:p
    U = rand;
    if U <= PI;
        GAM(i) = 1;
    end
end
Z = zeros(n,1);
for i = 1:n
    if Y(i) > 0 
        Z(i) = normlt(0,1,0);
    else
        Z(i) = normrt(0,1,0);
    end
end

%save in sparse matrix form (col1=rowID, col2=colID, col3=beta):
fid = fopen('beta.txt', 'a');
fprintf(fid, '%10d %10d %1.12e\n', [MCMC p 0]');    %store dimensions

if (nargin == 7)  %block updates
    %save number of neighbours):
    fidnumneigh = fopen('numneigh.txt', 'a');
    
    %%%% START MCMC
    for i = 1:MCMC
        [Z,beta,GAM,R,numneigh] = ...
            ZbetaGAM_update(X,Y,v,PI, Z,GAM, T, block, maxneigh);
            
        [rowGAM,colGAM] = find(GAM);
        fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta]');
        fprintf(fidnumneigh, '%1.6f ', numneigh);
        %fprintf(fidalpha, '%1.6f ', alpha);
    end
    fclose(fidnumneigh);

elseif (nargin == 5)    %single-components updates
    %save alpha:
    fidalpha = fopen('alpha.txt', 'a');

    %%%% START MCMC
    for i = 1:MCMC
        [Z,beta,GAM,R,alpha] = ZbetaGAM_update(X,Y,v,PI, Z,GAM, T);
        %[Z,beta,GAM,alpha] = ZbetaGAM_update_old(X,Y,v,PI, Z,GAM);
        
        [rowGAM,colGAM] = find(GAM);
        fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta]');
        fprintf(fidalpha, '%1.6f ', alpha);
    end
    fclose(fidalpha);
    
else
    error('Wrong # of arguments to probitVS');
end
fclose(fid);

% Manuela Zucknick, last update: 12-09-2006 (output alpha if nargin==5: 08-12-2007)