function logisticVS_tempMult(X,v,PI,Y,MCMC, temp,s,AE,burnin, block,K2,K1,update)
% LOGISTICVS_TEMPMULT Bayesian auxiliary variable model for logistic 
% regression using joint update to {z,lambda} with covariate set 
% uncertainty. Based on pseudocode A3 in Holmes and Held (2006).
% LOGISTICVS_TEMPMULT attempts to improve chain mixing by
% MCMCMC with tempered chains
%
% References: 
% Holmes, C. and Held, L. (2006) "Bayesian auxiliary variable models for 
% binary and multinomial regression", Bayesian Analysis 1:145-168
% Calvo F. (2005) "All-exchanges parallel tempering", The Journal of
% Chemical Physics 123:124106
%
% n = number of individuals 
% Y = nx1 column vector of binary variables
% X = nxp design matrix 
% v = prior covariance matrix for beta. v is a pxp matrix
%     It is assumed here that v is diagonal (for computations of v_gam)!
% PI = prior probability Pr(GAM_i = 1) (can be of size 1x1 or px1)
% MCMC = number of MCMC samples
% We are assuming that the prior mean for beta is the zero vector
%
% temp = vector of temperatures for tempered Markov chains (p(X)^(1/temp)),
%        first element has to be 1 (untempered chain)
% s = probability for swap attempt in each iteration -> 
%     Number of iterations before next swap attempt: x ~ Geo(s)=s(1-s)^x
%     Expected number of iterations before next swap attempt = (1-s)/s
% AE = if TRUE then all-exchange parallel tempering is done (Calvo 2005),
%      if FALSE, then only neighbouring chains are attempted to be swapped
% burnin = number of iterations the parallel chains are run uncoupled,
%          before exchange moves are attempted (must be burnin <= MCMC)
%          During burn-in, beta is saved for all chains
%
% optional:
% block = (sparse) pxp matrix with entries <>0, if the corresponding 
%         two genes should be updated together
% K2 = if > 0: scalar indicating up to how many second-order 
%      neighbours should be included (in addition to first-order nb.)
%      (option only for update = 'singleGibbs')
%      if <= 0: NO second-order neighbours are included
% K1 = if > 0: scalar indicating how many first-order neighbours should
%      be included max.
%      if <= 0: ALL first-order neighbours are included
% update = which method to update the gamma_i within the selected block,
%          one of the following: 'singleGibbs', 'joint'
% If block parameters are given, then GAM vector is updated by
% block proposal, otherwise by simple addition/deletion proposal dist.

numT = numel(temp);

n = size(X,1);
p = size(X,2);

if numel(PI)==1
    PI = ones(p,1).*PI;
end

%%%% INITIALISE RANDOM VARIABLES 
LAM = zeros(n,n,numT);
GAM = zeros(p,numT);
beta = cell(numT,1);
Z = zeros(n,numT);
for t = 1:numT
    LAM(:,:,t) = eye(n); 
    for i = 1:p
        U = rand;
        if U <= PI(i)
            GAM(i,t) = 1;
        end
    end
    for i = 1:n
        if Y(i) > 0 
            Z(i,t) = lefttrunclogis(0,1,0);
        else
            Z(i,t) = righttrunclogis(0,1,0);
        end
    end
end
logprob = zeros(numT,1);

if (burnin > 0)
  for t = 1:numT
    eval(strcat('fid',num2str(t),...
        '=fopen(strcat(''beta'',num2str(t),''.txt''),''a'');'))
    eval(strcat('fprintf(fid',num2str(t),...
        ',''%10d %10d %1.12e\n'', [burnin p 0]'');'))
    
    eval(strcat('fidprob',num2str(t),...
        '=fopen(strcat(''logprob'',num2str(t),''.txt''),''a'');'))
  end
end
if (MCMC > burnin)
    %save in sparse matrix form (col1=rowID, col2=colID, col3=beta):
    fid = fopen('beta.txt','a');
    fprintf(fid, '%10d %10d %1.12e\n', [(MCMC-burnin) p 0]'); %store dim.
    %save alpha and acceptance indicator:
    fidalpha = fopen('alpha.txt','a');
    fidindic = fopen('indic.txt','a');
    fidtA = fopen('tA.txt','a');
    fidtB = fopen('tB.txt','a');
end
    
%%%% START MCMC
if (nargin == 9)
  ad = zeros(numT,1);
  if (burnin > 0)
    for t = 1:numT
        %save addition/deletion alphas for all chains
        eval(strcat('fidad',num2str(t),...
        '=fopen(strcat(''ad'',num2str(t),''.txt''),''a'');'))
    end
  end
  
  for i = 1:MCMC
    %(1) Update beta, GAM and LAM, Z
    for t = 1:numT
        [betaTmp,GAM(:,t),ad(t),logpriorGAMbeta] = ...
        betaGAM_update_simple(LAM(:,:,t),Z(:,t),X,v,PI,GAM(:,t),temp(t));
        beta(t) = {betaTmp};
        [LAM(:,:,t),Z(:,t),logLAMZ] = ...
        LAMz_update(betaTmp,GAM(:,t),X,Y,temp(t)); 
 
        logprob(t) = logpriorGAMbeta + logLAMZ;
    end
    
    %(2) Attempt to exchange values between parallel chains
    if (i > burnin)
      if (~AE) %select neighbouring pair of chains at random for swap attempt    
        tA = randperm(numT-1);
        tA = tA(1);
        tB = tA+1;
        
        %Metropolis-Hastings acceptance for swapping GAM,beta,LAM and Z:
        alpha=min(1,exp((1/temp(tA)-1/temp(tB))*(logprob(tB)-logprob(tA))));
        fprintf(fidalpha, '%1.6f ', alpha);
    
      else     %all-exchange PT: compute all swap probs and draw accordingly
        alphas = zeros(numT-1);   %matrix
        for t1 = 1:(numT-1)
           for t2 = t1:(numT-1)
             alphas(t2,t1) = ...
             min(1,exp((1/temp(t1)-1/temp(t2))*(logprob(t2)-logprob(t1))));
           end
        end
        alphvec = [1 nonzeros(alphas)']; %first element for no swap
        probs = alphvec./sum(alphvec);
        
        %sample from probs
        Uni = rand(1);
        cumprob = [0 cumsum(probs)];    
            %%divide (0,1) into intervals of lengths prob(1),...,prob(k)
        indvec = max(find(Uni > cumprob));   
            %%assign index of the interval into which U falls
        if indvec==1    %first element selected -> no swap
            alpha = 0; 
            fprintf(fidalpha, '%1.6f ', alpha);
        else            %other element than first selected -> enforce swap
            alpha = 1;
            k = 1;
            eta = numT-1;
            delta = indvec - 1;
            if delta <= eta
                tA = k;         %column
                tB = delta + k; %row
            else
                while delta > eta
                    k = k + 1;
                    delta = delta - eta;
                    eta = eta - 1;
                end
                tA = k;
                tB = delta + k;
            end
            %%find temperatures corresponding to selected linear index
                  
            fprintf(fidalpha, '%1.6f ', alphvec(indvec));
        end 
      end
    
      indic = 0;
      Us = rand;
      if Us <= s
        U = rand;
        if U <= alpha
            betatmp = cell2mat(beta(tB));
            beta(tB) = beta(tA);
            beta(tA) = {betatmp};
            GAMtmp = GAM(:,tB);
            GAM(:,tB) = GAM(:,tA);
            GAM(:,tA) = GAMtmp;
            Ztmp = Z(:,tB);
            Z(:,tB) = Z(:,tA);
            Z(:,tA) = Ztmp;
            LAMtmp = LAM(:,:,tB);
            LAM(:,:,tB) = LAM(:,:,tA);
            LAM(:,:,tA) = LAMtmp;
            
            indic = 1;
        else
            tA = 0;
            tB = 0;
        end
      end
      fprintf(fidindic, '%1.6f ', indic);
      fprintf(fidtA, '%1.6f ', tA);
      fprintf(fidtB, '%1.6f ', tB);
      
      %save current beta and GAM values of chain of interest:
      [rowGAM,colGAM] = find(GAM(:,1));
      fprintf(fid,'%10d %10d %1.12e\n', ...
              [(i-burnin)*colGAM rowGAM cell2mat(beta(1))]');
    end
    
    if (i <= burnin)
      for t = 1:numT
        %save addition/deletion alphas for all chains
        eval(strcat('fprintf(fidad',num2str(t),',''%1.6f '',ad(t));'));
      
        %save beta and GAM values for all chains
        [rowGAM,colGAM] = find(GAM(:,t));
        eval(strcat('fprintf(fid',num2str(t),...
        ',''%10d %10d %1.12e\n'',[i*colGAM rowGAM cell2mat(beta(t))]'');'))
    
        %save (unnormalised) log posterior probs for all chains
        eval(strcat('fprintf(fidprob',num2str(t),',''%1.6f '',logprob(t));'));
      end
    end
    
  end  
  if (burnin > 0)
    for t = 1:numT
      %save addition/deletion alphas for both chains
      eval(strcat('fclose(fidad',num2str(t),');'))
    end
  end
 
elseif (nargin > 9)
  for i = 1:MCMC
    %(1) Update beta and GAM
    for t = 1:numT
      if strcmp(update,'joint')
      [betaTmp,GAM(:,t),numneigh,logpriorGAMbeta]= ...
            betaGAM_update_blockJoint(LAM(:,:,t),...
            Z(:,t),X,v,PI,GAM(:,t),block,K1,temp(t));
      elseif strcmp(update,'singleGibbs')  
      [betaTmp,GAM(:,t),numneigh,logpriorGAMbeta]= ...
            betaGAM_update_blockSingleGibbs(LAM(:,:,t),...
            Z(:,t),X,v,PI,GAM(:,t),block,K2,K1,temp(t));
      else
        error('Choice for parameter ''update'' not recognised');
      end
      beta(t) = {betaTmp};
      [LAM(:,:,t),Z(:,t),logLAMZ] = ...
      LAMz_update(betaTmp,GAM(:,t),X,Y,temp(t)); 
   
      logprob(t) = logpriorGAMbeta + logLAMZ;
    end
    
    %(2) Attempt to exchange beta and GAM values between parallel chains
    if (i > burnin)
      if (~AE) %select neighbouring pair of chains at random for swap attempt    
        tA = randperm(numT-1);
        tA = tA(1);
        tB = tA+1;
        
        %Metropolis-Hastings acceptance for swapping GAM,beta,LAM and Z:
        alpha=min(1,exp((1/temp(tA)-1/temp(tB))*(logprob(tB)-logprob(tA))));
        fprintf(fidalpha, '%1.6f ', alpha);
    
      else     %all-exchange PT: compute all swap probs and draw accordingly
        alphas = zeros(numT-1);   %matrix
        for t1 = 1:(numT-1)
           for t2 = t1:(numT-1)
             alphas(t2,t1) = ...
             min(1,exp((1/temp(t1)-1/temp(t2))*(logprob(t2)-logprob(t1))));
           end
        end
        alphvec = [1 nonzeros(alphas)']; %first element for no swap
        probs = alphvec./sum(alphvec);
        
        %sample from probs
        Uni = rand(1);
        cumprob = [0 cumsum(probs)];    
            %%divide (0,1) into intervals of lengths prob(1),...,prob(k)  
        indvec = max(find(Uni > cumprob));   
            %%assign index of the interval into which U falls
        if indvec==1    %first element selected -> no swap
            alpha = 0; 
            fprintf(fidalpha, '%1.6f ', alpha);
        else            %other element than first selected -> enforce swap
            alpha = 1;
            k = 0;
            eta = numT-1;
            delta = indvec;
            if delta <= eta
                tA = numT;
                tB = 1;
            else
                while delta > eta
                    k = k + 1;
                    delta = delta - eta;
                    eta = eta - 1;
                end
                tB = k + 1;
                tA = delta + k + 1; %"+1" for added row1 all made of zeros
            end
            %%find temperatures corresponding to selected linear index
                  
            fprintf(fidalpha, '%1.6f ', alphvec(indvec));
        end
      end  
    
      indic = 0;
      Us = rand;
      if Us <= s
        U = rand;
        if U <= alpha
            betatmp = cell2mat(beta(tB));
            beta(tB) = beta(tA);
            beta(tA) = {betatmp};
            GAMtmp = GAM(:,tB);
            GAM(:,tB) = GAM(:,tA);
            GAM(:,tA) = GAMtmp;
            Ztmp = Z(:,tB);
            Z(:,tB) = Z(:,tA);
            Z(:,tA) = Ztmp;
            LAMtmp = LAM(:,:,tB);
            LAM(:,:,tB) = LAM(:,:,tA);
            LAM(:,:,tA) = LAMtmp;
            
            indic = 1;
        else
            tA = 0;
            tB = 0;
        end
      end
      fprintf(fidindic, '%1.6f ', indic);
      fprintf(fidtA, '%1.6f ', tA);
      fprintf(fidtB, '%1.6f ', tB);
      
      %save current beta and GAM values of chain of interest:
      [rowGAM,colGAM] = find(GAM(:,1));
      fprintf(fid,'%10d %10d %1.12e\n', ...
              [(i-burnin)*colGAM rowGAM cell2mat(beta(1))]');
    end
    
    if (i <= burnin)
      for t = 1:numT
        %save beta and GAM values for all chains
        [rowGAM,colGAM] = find(GAM(:,t));
        eval(strcat('fprintf(fid',num2str(t),...
        ',''%10d %10d %1.12e\n'',[i*colGAM rowGAM cell2mat(beta(t))]'');'))
    
        %save (unnormalised) log posterior probs for all chains
        eval(strcat('fprintf(fidprob',num2str(t),',''%1.6f '',logprob(t));'));
      end
    end
  end  

else
  error('Wrong # of arguments to logisticVS_tempMult');
end

if (burnin > 0)
  for t = 1:numT
    eval(strcat('fclose(fid',num2str(t),');'));
    eval(strcat('fclose(fidprob',num2str(t),');'));
  end
end
if (MCMC > burnin)
    fclose(fid);
    fclose(fidalpha);
    fclose(fidindic);
    fclose(fidtA);
    fclose(fidtB);
end

% Manuela Zucknick, last updated: 16-03-2007
