function logisticVS(X, v, PI, Y, MCMC, block, K2, K1, update)
% LOGISTICVS Bayesian auxiliary variable model for logistic 
% regression using joint update to {z,lambda} with covariate set 
% uncertainty. Based on pseudocode A3 in Holmes and Held (2006)
%
% Reference: 
% Holmes, C. and Held, L. (2006) "Bayesian auxiliary variable models for 
% binary and multinomial regression", Bayesian Analysis 1:145-168
%
% n = number of individuals 
% Y = nx1 column vector of binary variables
% X = nxp design matrix 
% v = prior covariance matrix for beta. v is a pxp matrix
%     It is assumed here that v is diagonal (for computations of v_gam)!
% PI = prior probability Pr(GAM_i = 1) (can be of size 1x1 or px1)
% MCMC = number of MCMC samples
% We are assuming that the prior mean for beta is the zero vector
% temp = temperature, if > 1 then MCMCMC for GAM with tempered logisticVS 
%        is attempted
% optional:
% block = (sparse) pxp matrix with entries <>0, if the corresponding 
%         two genes should be updated together OR
%         string 'all' if full Gibbs sampling across all genes is intended
%         (option only for 'singleGibbs', K1&K2 are ignored in this case)
% K2 = if > 0: scalar indicating up to how many second-order 
%      neighbours should be included (in addition to first-order nb.)
%      (option only for update = 'singleGibbs')
%      if <= 0: NO second-order neighbours are included
% K1 = if > 0: scalar indicating how many first-order neighbours should
%      be included max.
%      if <= 0: ALL first-order neighbours are included
% update = which method to update the gamma_i within the selected block,
%          one of the following: 'singleGibbs', 'joint' and "randomNB'
%          (singleGibbs but with randomly selected neighbours)
% If parameter block (and optionally maxneigh and update) is given, then 
% the GAM vector is updated by a block proposal, 
% otherwise by an addition/deletion proposal.

nvars = nargin;
%if nargin==5 || nargin==7
%    temp = 1;
%    nvars = nvars + 1;
%end

n = size(X,1);
p = size(X,2);

if numel(PI)==1
    PI = ones(p,1).*PI;
end

%%%% INITIALISE RANDOM VARIABLES
LAM = eye(n);
GAM = zeros(p,1);
for i = 1:p
    U = rand;
    if U <= PI(i)
        GAM(i) = 1;
    end
end
Z = zeros(n,1); 
for i = 1:n
    if Y(i) > 0 
        Z(i) = lefttrunclogis(0,1,0);
    else
        Z(i) = righttrunclogis(0,1,0);
    end
end

%save in sparse matrix form (col1=rowID, col2=colID, col3=beta):
fid = fopen('beta.txt', 'a');
fprintf(fid, '%10d %10d %1.12e\n', [MCMC p 0]');    %store dimensions

if (nvars > 5)  %block updates
    if (strcmpi(block,'all') & ~strcmpi(update,'singleGibbs'))
    error('Option block = "all" is only available for update = "singleGibbs"');
    end
    
    maxn = 0;
    upd = 'singleGibbs';
    if (nvars == 8)
        maxn = maxneigh;
        upd = update;
    elseif (nvars == 7)
        maxn = maxneigh;
    end
    
    %save number of neighbours):
    fidnumneigh = fopen('numneigh.txt', 'a');
    
    %%%% START MCMC
    for i = 1:MCMC
        if strcmpi(update,'joint')
            [beta,GAM,numneigh] = ...
            betaGAM_update_blockJoint(LAM,Z, X,v,PI,GAM,block,K1);
        elseif strcmpi(update,'singleGibbs')  
            [beta,GAM,numneigh] = ...
            betaGAM_update_blockSingleGibbs(LAM,Z,X,v,PI,GAM,block,K2,K1);
        elseif strcmpi(update,'randomNB')  
            [beta,GAM,numneigh] = ...
            betaGAM_update_blockRand(LAM,Z,X,v,PI,GAM,block);
        else
            error('Choice for parameter ''update'' not recognised');
        end
        [LAM,Z] = LAMz_update(beta,GAM, X,Y);
        
        [rowGAM,colGAM] = find(GAM);
        fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta]');
        fprintf(fidnumneigh, '%1.6f ', numneigh);
    end
    fclose(fidnumneigh);

elseif (nvars == 5)    %single-components updates
    %save alpha:
    fidalpha = fopen('alpha.txt', 'a');

    %%%% START MCMC
    for i = 1:MCMC
        [beta,GAM,alpha] = betaGAM_update_simple(LAM,Z,X,v,PI,GAM);
        [LAM,Z] = LAMz_update(beta,GAM, X,Y);
    
        [rowGAM,colGAM] = find(GAM);
        fprintf(fid, '%10d %10d %1.12e\n', [i*colGAM rowGAM beta]');
        fprintf(fidalpha, '%1.6f ', alpha);
    end
    fclose(fidalpha);
    
else
    error('Wrong # of arguments to logisticVS');
end
fclose(fid);

% Manuela Zucknick, last update: 12-01-2008
