function [Z,beta,GAM, R,monitor] = ...
    ZbetaGAM_update(X,Y,v,PI, Z,GAM, T, block, maxneigh)
% ZBETAGAM_UPDATE Metropolis-Hastings update for beta, GAM and Z in 
% Bayesian auxiliary variable model for binary probit regression with 
% variable selection
% First update GAM with Metropois-hastings step, then update Z and beta.
% Help function for PROBITVS
%
% References:
% Holmes, C. and Held, L. (2006) "Bayesian auxiliary variable models for 
% binary and multinomial regression", Bayesian Analysis 1:145-168
% Lee, K.E. et al. (2003) "Gene selection: a Bayesian variable selection
% approach", Bioinformatics 19:90-97

numarg = nargin;

n = size(X,1);
p = size(X,2);

%(0) compute values corresponding to current GAM
Xgam = X(:,logical(GAM));
vgam = v(logical(GAM),logical(GAM));
vgam = vgam/T;
if sum(GAM) > n
    %inversion of nxn matrix:
    Vgam = vgam - vgam*Xgam'*inv(eye(n) + Xgam*vgam*Xgam')*Xgam*vgam;
else
    %inversion of dxd matrix (d=dim(GAM)):
    Vgam = inv(Xgam'*Xgam + inv(vgam));  
end
Sgam = Vgam*Xgam';
Bgam = Sgam*Z;

if numarg == 7
    %(1) proposal for GAM: Addition/deletion 
    %   (select covariate randomly and change GAM_i from 0 to 1 or 1 to 0)
    GAMstar = GAM;
    select = randperm(p);
    select = select(1);
    if GAM(select,1) == 0
        GAMstar(select,1) = 1;
        priorratio = PI(select)/(1-PI(select)); %prior(GAMstar)/prior(GAM)
    else
        GAMstar(select,1) = 0;
        priorratio = (1-PI(select))/PI(select);
    end
    
    %(2) compute values corresponding to proposed GAMstar
    Xstar = X(:,logical(GAMstar));
    vstar = v(logical(GAMstar),logical(GAMstar));
    vstar = vstar/T;
    if sum(GAMstar) > n
        %inversion of nxn matrix:
        Vstar=vstar-vstar*Xstar'*inv(eye(n)+Xstar*vstar*Xstar')*Xstar*vstar;
    else
        %inversion of dxd matrix (d=dim(GAMstar)):
        Vstar = inv(Xstar'*Xstar + inv(vstar));  
    end
    Sstar = Vstar*Xstar';
    Bstar = Sstar*Z;

    %(3) acceptance/rejection step
    detVratio = sqrt(det(Vstar)/det(Vgam));
    expBratio = exp((1/T)*0.5*Bstar'*inv(Vstar)*Bstar - ...
                    (1/T)*0.5*Bgam' *inv(Vgam) *Bgam);
    detvratio = sqrt(det(vgam)/det(vstar));
    alpha = min(1, priorratio*detVratio*expBratio*detvratio);
    U = rand;
    if U < alpha
        GAM = GAMstar;
        Xgam = Xstar;
        Vgam = Vstar;
        vgam = vstar;
        Sgam = Sstar;
        Bgam = Bstar;
    end
    
    monitor = alpha;
    
elseif numarg == 9
    %(1) proposal for GAM: Block proposal
    %   (select gamma_i at random, find 'mates' and for all of these,
    %   draw from p(GAM_i|Z) [Lee et al. (2003)]: draw by computing density
    %   for both possible values (1 and 0). One of both values is available 
    %   from GAM -> only need to compute the other one = GAMtry)
    
    Zprecgam = (eye(n) - Xgam*Vgam*Xgam')/T;
    detZprecgam = det(Zprecgam);
    Pgam = sqrt(detZprecgam)*exp(-0.5*(Z'*Zprecgam*Z)); 
    %p(Z|GAM)=MVN(0,inv(Zprec))

    %find variables to update together with GAM(select) (= block)
    select = randperm(p);
    select = select(1);

    neighbour = unique([select, find(block(select,:))]);   %1st order nb.

    if (maxneigh > 0)
        neighbour2 = 0;
        for k = 1:numel(neighbour)  %1st order nb of nb1 = 2nd order nb
            neighbour2 = [neighbour2, find(block(neighbour(1,k),:))];
        end
        neighbour2 = neighbour2(2:numel(neighbour2));   %remove 0
        neighbour2 = unique(neighbour2);

        %%select maxneigh genes for updating among all second-order nb's
        numneigh = numel(neighbour2);
        selectneigh = randperm(numneigh);
        selectneigh = neighbour2(selectneigh);
        maxneigh = min(maxneigh, numneigh);
        selectneigh = selectneigh(1:maxneigh);  
        neighbour = unique([neighbour, selectneigh]);
    end  

    numneigh = numel(neighbour);
    GAMstar = GAM;
    for index = 1:numneigh
        i = neighbour(index);
    
        GAMtry = GAM;
        if GAM(i) == 0
            GAMtry(i) = 1;
        else
            GAMtry(i) = 0;
        end
    
        Xtry = X(:,logical(GAMtry));
        vtry = v(logical(GAMtry),logical(GAMtry))/T;
        if sum(GAMtry) > n
            %inversion of nxn matrix:
            Vtry = vtry - vtry*Xtry'*inv(eye(n)+Xtry*vtry*Xtry')*Xtry*vtry;
        else
            %inversion of dxd matrix (d=dim(GAMtry)):
            Vtry = inv(Xtry'*Xtry + inv(vtry));  
        end
        Zprectry = (eye(n) - Xtry*Vtry*Xtry')/T;
        detZprectry = det(Zprectry);
        Ptry = sqrt(detZprectry)*exp(-0.5*(Z'*Zprectry*Z)); 
        %p(Z|GAM)=MVN(0,inv(Zprec))
    
        probTRY = Ptry * PI(i)^GAMtry(i)*(1-PI(i))^(1-GAMtry(i));
        probGAM = Pgam * PI(i)^GAM(i)*(1-PI(i))^(1-GAM(i));
    
        if probTRY==0 && probGAM==0  %b/c comp. problems with precision
            if GAMtry(i)==0                 %-> favour more sparse GAM
                GAMstar(i) = GAMtry(i);     %else: GAMstar(i) = GAM(i)
            end
        else
            U = rand;
            if U <= probTRY/(probGAM + probTRY)
                GAMstar(i) = GAMtry(i);     %else: GAMstar(i) = GAM(i)
            end
        end
    end
    
    %(2) compute values for dist. of beta corresponding to proposed GAMstar
    Xstar = X(:,logical(GAMstar));
    vstar = v(logical(GAMstar),logical(GAMstar))/T;
    if sum(GAMstar) > n
        %inversion of nxn matrix:
        Vstar = vstar - vstar*Xstar'*inv(eye(n) + Xstar*vstar*Xstar')*Xstar*vstar;
    else
        %inversion of dxd matrix (d=dim(GAM)):
        Vstar = inv(Xstar'*Xstar + inv(vstar));  
    end
    Sstar = Vstar*Xstar';
    Bstar = Sstar*Z;

    %(3) acceptance/rejection step not needed, since alpha = 1 always
    alpha = 1;
    GAM = GAMstar;
    Xgam = Xstar;
    Vgam = Vstar;
    vgam = vstar;
    Sgam = Sstar;
    Bgam = Bstar;
    
    monitor = numneigh;

else
  error('Wrong # of arguments to ZbetaGAM_update');
end
    
%(4) compute beta and Z for new GAM
H = zeros(n,1);
W = zeros(n,1);
Q = zeros(n,1);
for j = 1:n
    H(j) = Xgam(j,:)*Sgam(:,j);
    W(j) = H(j)/(1 - H(j));
    Q(j) = (W(j) + 1)*T;
    
    zold = Z(j);
    m = Xgam(j,:)*Bgam;
    m = m - W(j)*(Z(j) - m);
    if Y(j) > 0 
        Z(j) = normlt(m,sqrt(Q(j)),0);
    else
        Z(j) = normrt(m,sqrt(Q(j)),0);
    end
    
    Bgam  = Bgam  + (Z(j) - zold)* Sgam(:,j);  %distribution of beta
end

Tau = randn(sum(GAM),1);
L = chol(Vgam*T)';   %lower triangular Cholesky factorisation matrix
beta = Bgam + L*Tau;
R = Z - Xgam*beta;

% Manuela Zucknick, last update: 20-09-06